/*
 * Decompiled with CFR 0.152.
 */
package org.apache.bookkeeper.statelib.impl.rocksdb.checkpoint;

import com.google.common.collect.Lists;
import com.google.common.collect.Sets;
import com.google.common.io.Files;
import com.google.common.io.MoreFiles;
import com.google.common.io.RecursiveDeleteOption;
import com.google.protobuf.UnsafeByteOperations;
import java.io.File;
import java.io.IOException;
import java.io.OutputStream;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.util.ArrayList;
import java.util.HashSet;
import java.util.List;
import java.util.UUID;
import org.apache.bookkeeper.statelib.api.checkpoint.CheckpointStore;
import org.apache.bookkeeper.statelib.api.exceptions.StateStoreException;
import org.apache.bookkeeper.statelib.impl.rocksdb.RocksUtils;
import org.apache.bookkeeper.stream.proto.kv.store.CheckpointMetadata;
import org.rocksdb.Checkpoint;
import org.rocksdb.RocksDBException;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class RocksdbCheckpointTask {
    private static final Logger log = LoggerFactory.getLogger(RocksdbCheckpointTask.class);
    private final String dbName;
    private final Checkpoint checkpoint;
    private final File checkpointDir;
    private final CheckpointStore checkpointStore;
    private final String dbPrefix;
    private final boolean removeLocalCheckpointAfterSuccessfulCheckpoint;
    private final boolean removeRemoteCheckpointsAfterSuccessfulCheckpoint;
    private InjectedError<String> injectedError = checkpointId -> {};

    public RocksdbCheckpointTask(String dbName, Checkpoint checkpoint, File checkpointDir, CheckpointStore checkpointStore, boolean removeLocalCheckpoint, boolean removeRemoteCheckpoints) {
        this.dbName = dbName;
        this.checkpoint = checkpoint;
        this.checkpointDir = checkpointDir;
        this.checkpointStore = checkpointStore;
        this.dbPrefix = String.format("%s", dbName);
        this.removeLocalCheckpointAfterSuccessfulCheckpoint = removeLocalCheckpoint;
        this.removeRemoteCheckpointsAfterSuccessfulCheckpoint = removeRemoteCheckpoints;
    }

    public void setInjectedError(InjectedError<String> injectedError) {
        this.injectedError = injectedError;
    }

    /*
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    public String checkpoint(byte[] txid) throws StateStoreException {
        String string;
        String checkpointId = UUID.randomUUID().toString();
        File tempDir = new File(this.checkpointDir, checkpointId);
        log.info("Create a local checkpoint of state store {} at {}", (Object)this.dbName, (Object)tempDir);
        try {
            String sstsPath;
            try {
                this.checkpoint.createCheckpoint(tempDir.getAbsolutePath());
            }
            catch (RocksDBException e) {
                throw new StateStoreException("Failed to create a checkpoint at " + tempDir, e);
            }
            String remoteCheckpointPath = RocksUtils.getDestCheckpointPath(this.dbPrefix, checkpointId);
            if (!this.checkpointStore.fileExists(remoteCheckpointPath)) {
                this.checkpointStore.createDirectories(remoteCheckpointPath);
            }
            if (!this.checkpointStore.fileExists(sstsPath = RocksUtils.getDestSstsPath(this.dbPrefix))) {
                this.checkpointStore.createDirectories(sstsPath);
            }
            this.injectedError.accept(checkpointId);
            List<File> filesToCopy = this.getFilesToCopy(tempDir);
            this.copyFilesToDest(checkpointId, filesToCopy);
            this.finalizeCopyFiles(checkpointId, filesToCopy);
            this.finalizeCheckpoint(checkpointId, tempDir, txid);
            if (this.removeRemoteCheckpointsAfterSuccessfulCheckpoint) {
                this.cleanupRemoteCheckpoints(tempDir, checkpointId);
            }
            string = checkpointId;
            if (!this.removeLocalCheckpointAfterSuccessfulCheckpoint) return string;
            if (!tempDir.exists()) return string;
        }
        catch (IOException ioe) {
            try {
                log.error("Failed to checkpoint db {} to dir {}", new Object[]{this.dbName, tempDir, ioe});
                throw new StateStoreException("Failed to checkpoint db " + this.dbName + " to dir " + tempDir, ioe);
            }
            catch (Throwable throwable) {
                if (!this.removeLocalCheckpointAfterSuccessfulCheckpoint) throw throwable;
                if (!tempDir.exists()) throw throwable;
                try {
                    MoreFiles.deleteRecursively((Path)Paths.get(tempDir.getAbsolutePath(), new String[0]), (RecursiveDeleteOption[])new RecursiveDeleteOption[]{RecursiveDeleteOption.ALLOW_INSECURE});
                    throw throwable;
                }
                catch (IOException ioe2) {
                    log.warn("Failed to remove temporary checkpoint dir {}", (Object)tempDir, (Object)ioe2);
                }
                throw throwable;
            }
        }
        try {
            MoreFiles.deleteRecursively((Path)Paths.get(tempDir.getAbsolutePath(), new String[0]), (RecursiveDeleteOption[])new RecursiveDeleteOption[]{RecursiveDeleteOption.ALLOW_INSECURE});
            return string;
        }
        catch (IOException ioe) {
            log.warn("Failed to remove temporary checkpoint dir {}", (Object)tempDir, (Object)ioe);
        }
        return string;
    }

    private List<File> getFilesToCopy(File checkpointedDir) throws IOException {
        File[] files = checkpointedDir.listFiles();
        ArrayList fileToCopy = Lists.newArrayListWithExpectedSize((int)files.length);
        for (File file : files) {
            if (RocksUtils.isSstFile(file)) {
                String destSstPath = RocksUtils.getDestSstPath(this.dbPrefix, file);
                if (this.checkpointStore.fileExists(destSstPath)) continue;
                fileToCopy.add(file);
                continue;
            }
            fileToCopy.add(file);
        }
        return fileToCopy;
    }

    private void copyFilesToDest(String checkpointId, List<File> files) throws IOException {
        for (File file : files) {
            this.copyFileToDest(checkpointId, file);
        }
    }

    private void copyFileToDest(String checkpointId, File file) throws IOException {
        String destPath = RocksUtils.getDestPath(this.dbPrefix, checkpointId, file);
        try (OutputStream os = this.checkpointStore.openOutputStream(destPath);){
            Files.copy((File)file, (OutputStream)os);
        }
    }

    private void finalizeCopyFiles(String checkpointId, List<File> files) throws IOException {
        for (File file : files) {
            if (!RocksUtils.isSstFile(file)) continue;
            String destSstTempPath = RocksUtils.getDestTempSstPath(this.dbPrefix, checkpointId, file);
            String destSstPath = RocksUtils.getDestSstPath(this.dbPrefix, file);
            this.checkpointStore.rename(destSstTempPath, destSstPath);
        }
    }

    private void finalizeCheckpoint(String checkpointId, File checkpointedDir, byte[] txid) throws IOException {
        File[] files = checkpointedDir.listFiles();
        CheckpointMetadata.Builder metadataBuilder = CheckpointMetadata.newBuilder();
        for (File file : files) {
            metadataBuilder.addFiles(file.getName());
        }
        if (null != txid) {
            metadataBuilder.setTxid(UnsafeByteOperations.unsafeWrap((byte[])txid));
        }
        metadataBuilder.setCreatedAt(System.currentTimeMillis());
        String destCheckpointPath = RocksUtils.getDestCheckpointMetadataPath(this.dbPrefix, checkpointId);
        try (OutputStream os = this.checkpointStore.openOutputStream(destCheckpointPath);){
            os.write(metadataBuilder.build().toByteArray());
        }
    }

    private void cleanupRemoteCheckpoints(File checkpointedDir, String checkpointToExclude) throws IOException {
        String[] checkpointedFiles;
        String checkpointsPath = RocksUtils.getDestCheckpointsPath(this.dbPrefix);
        List<String> checkpoints = this.checkpointStore.listFiles(checkpointsPath);
        for (String checkpoint : checkpoints) {
            if (checkpoint.equals(checkpointToExclude)) continue;
            String string = RocksUtils.getDestCheckpointPath(this.dbPrefix, checkpoint);
            this.checkpointStore.deleteRecursively(string);
            log.info("Delete remote checkpoint {} from checkpoint store at {}", (Object)checkpoint, (Object)string);
        }
        HashSet checkpointedFileSet = Sets.newHashSet();
        for (String file : checkpointedFiles = checkpointedDir.list()) {
            checkpointedFileSet.add(file);
        }
        List<String> list = this.checkpointStore.listFiles(RocksUtils.getDestSstsPath(this.dbPrefix));
        for (String sst : list) {
            if (checkpointedFileSet.contains(sst)) continue;
            this.checkpointStore.delete(RocksUtils.getDestSstPath(this.dbPrefix, sst));
        }
    }

    @FunctionalInterface
    public static interface InjectedError<T> {
        public void accept(T var1) throws IOException;
    }
}

